﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;

namespace DiskCopy2Dsk
{
    public partial class Form1 : Form
    {
        string outputDirectory;

        public Form1()
        {
            InitializeComponent();
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            outputDirectory = Environment.GetFolderPath(Environment.SpecialFolder.Personal);
            outputDirectoryTextBox.Text = outputDirectory;
        }

        private void button2_Click(object sender, EventArgs e)
        {
            OpenFileDialog ofg = new OpenFileDialog();
            ofg.Title = "Select DiskCopy 4.2 image files to convert";
            ofg.Filter = "All files|*.*";
            ofg.Multiselect = true;

            if (ofg.ShowDialog() == DialogResult.OK)
            {
                foreach (string filename in ofg.FileNames)
                {
                    try
                    {
                        FileStream rfs = new FileStream(filename, FileMode.Open, FileAccess.Read, FileShare.ReadWrite);
                        BinaryReader readFile = new BinaryReader(rfs);

                        // ensure the file is at least 400K, and has the DiskCopy 4.2 magic bytes
                        if (rfs.Length < 400 * 1024)
                        {
                            MessageBox.Show(Path.GetFileName(filename) + " is not a DiskCopy 4.2 image file", "Error");
                            return;
                        }

                        byte[] bytes = readFile.ReadBytes(0x54);
                        if (bytes[0x52] != 0x01 || bytes[0x53] != 0x00)
                        {
                            MessageBox.Show(Path.GetFileName(filename) + " is not a DiskCopy 4.2 image file", "Error");
                            return;
                        }

                        // ensure the embedded data size is 400, 800, or 1440 K
                        UInt32 dataSize = (UInt32)(bytes[0x40] * 16777216 + bytes[0x41] * 65536 + bytes[0x42] * 256 + bytes[0x43]);
                        if (dataSize != 400 * 1024 && dataSize != 800 * 1024 && dataSize != 1440 * 1024)
                        {
                            MessageBox.Show(Path.GetFileName(filename) + " is an unsupported image size", "Error");
                            return;
                        }

                        // get the embedded image data, and write it out as a new file with a .dsk extension
                        byte[] imageData = readFile.ReadBytes((int)dataSize);
                        string saveFilename = outputDirectory + "\\" + Path.GetFileNameWithoutExtension(filename) + ".dsk";

                        FileStream wfs = new FileStream(saveFilename, FileMode.Create, FileAccess.Write, FileShare.ReadWrite);
                        BinaryWriter writeFile = new BinaryWriter(wfs);

                        writeFile.Write(imageData);

                        writeFile.Close();
                        readFile.Close();

                        rfs.Close();
                        wfs.Close();
                    }
                    catch (Exception)
                    {
                        MessageBox.Show("An error occurred while converting " + Path.GetFileName(filename), "Error");
                    }

                    MessageBox.Show("Success!", "DiskCopy2Dsk Result");
                }
            }
        }

        private void button1_Click(object sender, EventArgs e)
        {
            FolderBrowserDialog dialog = new FolderBrowserDialog();

            if (dialog.ShowDialog() == DialogResult.OK)
            {
                outputDirectory = dialog.SelectedPath;
                outputDirectoryTextBox.Text = outputDirectory;
            }
        }
    }
}
